/**************************************************************************//**
*
*	\file blueboxlib.h
*
*	\brief blueboxlib.cpp header file.
*
*	This module contains all the function libraries implemented to use and
*	configure all the Soltec Bluebox RFID readers. The library is available in
*	the following formats:
*	- Win32 DLL (Soltec provides the BLUEBOXLib.lib stub for Microsoft Visual
*		C++ 6.0)
*
******************************************************************************/

#ifndef _BLUEBOX_LIB_H
#define _BLUEBOX_LIB_H

#ifdef __cplusplus
extern "C" {
#endif

//-----------------------------------------------------------------------------
//--- INCLUDE FILES -----------------------------------------------------------
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
//--- DEFINITIONS -------------------------------------------------------------
//-----------------------------------------------------------------------------

/******************************************************************************
* The following ifdef block is the standard way of creating macros which make 
* exporting from a DLL simpler. All files within this DLL are compiled with the 
* BLUEBOXLIB_EXPORTS symbol defined on the command line. This symbol should not 
* be defined on any project that uses this DLL. This way any other project
* whose source files include this file see BLUEBOXLib_API functions as being
* imported from a DLL, wheras this DLL sees symbols defined with this macro as
* being exported.
******************************************************************************/
#ifdef BLUEBOXLIB_EXPORTS
#define BLUEBOXLib_API __declspec(dllexport)
#else
#define BLUEBOXLib_API __declspec(dllimport)
#endif

/// Handle type.
typedef int BLUEBOX_Handle;

/// Error codes enum.
typedef enum BLUEBOX_ErrorCodes
{
	/// Operation completed succesfully.
	BLUEBOX_StatusOk = 0,
	/// Initialization error.
	BLUEBOX_InitError = -1,
	/// Invalid handle.
	BLUEBOX_InvalidHandle = -2,
	/// Invalid channel.
	BLUEBOX_InvalidChannel = -3,
	/// Invalid parameters.
	BLUEBOX_InvalidParams = -4,
	/// Generic error.
	BLUEBOX_GenericError = -5,
	/// Timeout error.
	BLUEBOX_TimeoutError = -6,
	/// Communication error.
	BLUEBOX_CommunicationError = -7,
	/// Connection error.
	BLUEBOX_ConnectionError = -8,
	/// Allocation memory error.
	BLUEBOX_MemoryError = -9,
	/// Invalid command.
	BLUEBOX_InvalidCommand = -10,
	/// Tag not found.
	BLUEBOX_TagNotFound = -11,
	/// Tag error.
	BLUEBOX_TagError = -12,
	/// Notify trigger allocation error.
	BLUEBOX_AllocationError = -13,
	/// File error.
	BLUEBOX_FileError = -14,
	/// Registration not found.
	BLUEBOX_RegistrationNotFound = -15

} BLUEBOX_ErrorCodes;

/// Input enum.
typedef enum BLUEBOX_Input
{
	/// No input information.
	BLUEBOX_NOINPUT = 0,
	/// Input 1.
	BLUEBOX_INPUT_1 = 1,
	/// Input 2.
	BLUEBOX_INPUT_2 = 2

} BLUEBOX_Input;

/// Output enum.
typedef enum BLUEBOX_Output
{
	/// Output 1.
	BLUEBOX_OUTPUT_1 = 1,
	/// Output 2.
	BLUEBOX_OUTPUT_2 = 2

} BLUEBOX_Output;

/// Antenna enum.
typedef enum BLUEBOX_Antenna
{
	/// No antenna information.
	BLUEBOX_NOANT = 0,
	/// Antenna n. 1.
	BLUEBOX_ANT_1 = 1,
	/// Antenna n. 2.
	BLUEBOX_ANT_2 = 2,
	/// Antenna n. 3.
	BLUEBOX_ANT_3 = 3,
	/// Antenna n. 4.
	BLUEBOX_ANT_4 = 4

} BLUEBOX_Antenna;

/// Tag type enum.
typedef enum BLUEBOX_TagType
{
	/// Undefined tag.
	BLUEBOX_UNDEFINED = 0,
	/// BLUEBOX SHORT.
	BLUEBOX_SHORT = 1,
	/// BLUEBOX MEDIUM.
	BLUEBOX_MEDIUM = 2,
	/// BLUEBOX LARGE.
	BLUEBOX_LARGE = 3,
	/// Q5.
	BLUEBOX_Q5 = 4,
	/// HITAG S 256.
	BLUEBOX_HITAG_S256 = 5,
	/// HITAG S 2048.
	BLUEBOX_HITAG_S2048 = 6,
	/// TITAN
	BLUEBOX_TITAN = 7,
	/// ISO 14443A
	BLUEBOX_ISO14443A = 8,
	/// MIFARE 1k
	BLUEBOX_MIFARE_1k = 9,
	/// MIFARE 4k
	BLUEBOX_MIFARE_4k = 10,
	/// MIFARE ULTRALIGHT
	BLUEBOX_MIFARE_UL = 11,
	/// ISO 15693
	BLUEBOX_ISO15693 = 12,
	/// I CODE 2
	BLUEBOX_ICODE2 = 13,
	/// TAG-IT HF-I
	BLUEBOX_TAG_IT_HF_I = 14,
	/// EM4035
	BLUEBOX_EM4035 = 15,
	/// LRI 64/512
	BLUEBOX_LRI_64_512 = 16,
	/// MB89R118
	BLUEBOX_MB89R118 = 17,
	/// ISO 14443B
	BLUEBOX_ISO14443B = 18,
	/// SR176
	BLUEBOX_SR176 = 19,
	/// ISO 18000-6B
	BLUEBOX_ISO18K6B = 20,
	/// ISO 18000-6C
	BLUEBOX_ISO18K6C = 21,
	/// ACTIVE
	BLUEBOX_ACTIVE = 22,
	/// EM4305
	BLUEBOX_EM4305 = 23,
	/// T5557
	BLUEBOX_T5557 = 24,
	/// I CODE SLI-S
	BLUEBOX_ICODE_SLI_S = 25,
	/// Hitag 1
	BLUEBOX_HITAG_1 = 26,
	/// MIFARE Mini
	BLUEBOX_MIFARE_MINI = 27,
	/// MIFARE Desfire
	BLUEBOX_MIFARE_DESFIRE = 28,
	/// MIFARE 7-byte UID 2k
	BLUEBOX_MIFARE_7BUID_2k = 29,
	/// MIFARE 7-byte UID 4k
	BLUEBOX_MIFARE_7BUID_4k = 30,
	/// MIFARE Plus 2k
	BLUEBOX_MIFARE_PLUS_2k = 31,
	/// MIFARE Plus 4k
	BLUEBOX_MIFARE_PLUS_4k = 32,
	/// SRI 512
	BLUEBOX_SRI512 = 33,
	/// JCOS
	BLUEBOX_JCOS = 34,
	/// PICOPASS
	BLUEBOX_PICOPASS = 35,
	/// Hitag 2
	BLUEBOX_HITAG_2 = 36,
	/// IBM JCOP
	BLUEBOX_JCOP = 37,
	/// IBM JCOP31
	BLUEBOX_JCOP31 = 38,
	/// IBM JCOP41
	BLUEBOX_JCOP41 = 39

} BLUEBOX_TagType;
 
/// Tag identification struct.
typedef struct BLUEBOX_Tag
{
	/// Tag type.
	BLUEBOX_TagType TagType;
	/// Tag ID.
	unsigned char *Id;
	/// The length of the ID.
	int Length;
	/// The antenna used to identify the tag.
	BLUEBOX_Antenna Antenna;
	// The input which activates the identification procedure.
	BLUEBOX_Input Input;

} BLUEBOX_Tag;

/// EM4305 tag's ID size in bytes.
#define	BLUEBOX_EM4305_ID_SIZE			(4)
/// T5557 tag's ID size in bytes.
#define	BLUEBOX_T5557_ID_SIZE			(8)
/// Q5 tag's ID size in bytes.
#define	BLUEBOX_Q5_ID_SIZE				(5)
/// HITAG 1 tag's ID size in bytes.
#define	BLUEBOX_HITAG1_ID_SIZE			(4)
/// HITAG 1 tag's memory page size in bytes.
#define	BLUEBOX_HITAG1_PAGE_SIZE		(4)
/// HITAG 2 tag's ID size in bytes.
#define	BLUEBOX_HITAG2_ID_SIZE			(4)
/// HITAG 2 tag's memory page size in bytes.
#define	BLUEBOX_HITAG2_PAGE_SIZE		(4)
/// HITAG 2 tag's password size in bytes.
#define	BLUEBOX_HITAG2_PASSWORD_SIZE	(4)
/// HITAG S tag's ID size in bytes.
#define	BLUEBOX_HITAGS_ID_SIZE			(4)
/// HITAG S tag's memory page size in bytes.
#define	BLUEBOX_HITAGS_PAGE_SIZE		(4)
/// TITAN tag's ID size in bytes.
#define	BLUEBOX_TITAN_ID_SIZE			(8)
/// TITAN tag's password size in bytes.
#define	BLUEBOX_TITAN_PASSWORD_SIZE		(4)
/// TITAN tag's memory page size in bytes.
#define	BLUEBOX_TITAN_PAGE_SIZE			(4)

/// ISO 15693 tag's UID size in bytes.
#define	BLUEBOX_ISO15693_UID_SIZE		(8)
/// ICODE SLI-S tag's random number size in bytes.
#define	BLUEBOX_ICODE_SLI_S_RND_SIZE	(2)
/// ICODE SLI-S tag's password size in bytes.
#define	BLUEBOX_ICODE_SLI_S_PWD_SIZE	(4)

/// ICODE SLI-S tag's password identifier.
typedef enum BLUEBOX_ICODE_SLI_S_PasswordIdentifier
{
	/// Read.
	BLUEBOX_ICODE_SLI_S_PWD_READ = 0x01,
	/// Write.
	BLUEBOX_ICODE_SLI_S_PWD_WRITE = 0x02,
	/// Privacy.
	BLUEBOX_ICODE_SLI_S_PWD_PRIVACY = 0x04,
	/// Destroy SLI-S.
	BLUEBOX_ICODE_SLI_S_PWD_DESTROY_SLI_S = 0x08,
	/// EAS.
	BLUEBOX_ICODE_SLI_S_PWD_EAS = 0x10

} BLUEBOX_ICODE_SLI_S_PasswordIdentifier;

/// ICODE SLI-S tag's protection status.
typedef enum BLUEBOX_ICODE_SLI_S_ProtectionStatus
{
	/// 32-bit / 64-bit Password Protection: Public.
	BLUEBOX_ICODE_SLI_S_PROTECT_PUBLIC = 0x00,
	/// 32-bit Password Protection: Read and Write protected by the Read password.
	BLUEBOX_ICODE_SLI_S_PROTECT_READ_AND_WRITE_BY_READ_PWD = 0x01,
	/// 32-bit Password Protection: Write protected by the Write password.
	BLUEBOX_ICODE_SLI_S_PROTECT_WRITE_BY_WRITE_PWD = 0x10,
	/// 32-bit Password Protection: Read protected by the Read password and Write protected by the Write password.
	BLUEBOX_ICODE_SLI_S_PROTECT_READ_BY_READ_PWD_AND_WRITE_BY_WRITE_PWD = 0x11,
	/// 64-bit Password Protection: Read and Write protected by the Read plus Write password.
	BLUEBOX_ICODE_SLI_S_PROTECT_READ_AND_WRITE_BY_READ_AND_WRITE_PWD = 0x01,
	/// 64-bit Password Protection: Write protected by the Read plus Write password.
	BLUEBOX_ICODE_SLI_S_PROTECT_WRITE_BY_READ_AND_WRITE_PWD = 0x10,

} BLUEBOX_ICODE_SLI_S_ProtectionStatus;

/// ICODE SLI-S tag's block protection status.
typedef struct BLUEBOX_ICODE_SLI_S_BlockProtectionStatus
{
	/// Lock bit.
	int LockBit;
	/// Read password protected.
	int ReadPasswordProtected;
	/// Write password protected.
	int WritePasswordProtected;
	/// Page protection lock.
	int PageProtectionLock;

} BLUEBOX_ICODE_SLI_S_BlockProtectionStatus;

/// MIFARE Mini tag's UID size in bytes.
#define	BLUEBOX_MIFARE_MINI_UID_SIZE	(4)
/// MIFARE 1k tag's UID size in bytes.
#define	BLUEBOX_MIFARE_1k_UID_SIZE		(4)
/// MIFARE 1k tag's memory block size in bytes.
#define	BLUEBOX_MIFARE_1k_BLOCK_SIZE	(16)
/// MIFARE 4k tag's UID size in bytes.
#define	BLUEBOX_MIFARE_4k_UID_SIZE		(4)
/// MIFARE 4k tag's memory block size in bytes.
#define	BLUEBOX_MIFARE_4k_BLOCK_SIZE	(16)
/// MIFARE Ultralight tag's UID size in bytes.
#define	BLUEBOX_MIFARE_UL_UID_SIZE		(7)
/// MIFARE Ultralight tag's memory block size in bytes.
#define	BLUEBOX_MIFARE_UL_BLOCK_SIZE	(4)
/// MIFARE tag's key size in bytes.
#define	BLUEBOX_MIFARE_KEY_SIZE			(6)
/// MIFARE Desfire tag's UID size in bytes.
#define	BLUEBOX_MIFARE_DESFIRE_UID_SIZE	(7)
/// MIFARE 2k 7-bytes UID tag's UID size in bytes.
#define	BLUEBOX_MIFARE_7BUID_2k_UID_SIZE	(7)
/// MIFARE 2k 7-bytes UID tag's memory block size in bytes.
#define	BLUEBOX_MIFARE_7BUID_2k_BLOCK_SIZE	(16)
/// MIFARE 4k 7-bytes UID tag's UID size in bytes.
#define	BLUEBOX_MIFARE_7BUID_4k_UID_SIZE	(7)
/// MIFARE 4k 7-bytes UID tag's memory block size in bytes.
#define	BLUEBOX_MIFARE_7BUID_4k_BLOCK_SIZE	(16)
/// MIFARE Plus 2k tag's UID size in bytes.
#define BLUEBOX_MIFARE_PLUS_2k_UID_SIZE	(7)
/// MIFARE Plus 4k tag's UID size in bytes.
#define BLUEBOX_MIFARE_PLUS_4k_UID_SIZE	(7)
/// NTAG21x tag's UID size in bytes.
#define	BLUEBOX_NTAG21x_UID_SIZE		(7)
/// NTAG21x tag's memory block size in bytes.
#define	BLUEBOX_NTAG21x_BLOCK_SIZE		(4)
/// SR 176 tag's UID size in bytes.
#define	BLUEBOX_SR176_UID_SIZE			(8)
/// SR 176 tag's memory block size in bytes.
#define	BLUEBOX_SR176_BLOCK_SIZE		(2)
/// SRI 512 tag's UID size in bytes.
#define	BLUEBOX_SRI512_UID_SIZE			(8)
/// Jcos tag's UID size in bytes.
#define BLUEBOX_JCOS_UID_SIZE			(8)
/// IBM JCOP tag's UID size in bytes.
#define	BLUEBOX_JCOP_UID_SIZE			(7)
/// IBM JCOP31 tag's UID size in bytes.
#define	BLUEBOX_JCOP31_UID_SIZE			(4)
/// IBM JCOP41 tag's UID size in bytes.
#define	BLUEBOX_JCOP41_UID_SIZE			(4)

/// MIFARE key to use to access to tag's memory.
typedef enum BLUEBOX_MIFARE_Key
{
	/// Key A.
	BLUEBOX_MIFARE_KEY_A = 0,
	/// Key B.
	BLUEBOX_MIFARE_KEY_B = 1

} BLUEBOX_MIFARE_Key;

/// PICOPASS tag's UID size in bytes.
#define	BLUEBOX_PICOPASS_UID_SIZE	(8)

/// ISO 18000-6B tag's UID size in bytes.
#define	BLUEBOX_ISO18K6B_UID_SIZE		(8)
/// ISO 18000-6B tag's memory block size in bytes.
#define	BLUEBOX_ISO18K6B_BLOCK_SIZE		(8)
/// ISO 18000-6C tag's UID size in bytes.
#define	BLUEBOX_ISO18K6C_UID_SIZE		(66)
/// ISO 18000-6C tag's memory block size in bytes.
#define	BLUEBOX_ISO18K6C_BLOCK_SIZE		(2)
/// ISO 18000-6C tag's access password size in bytes.
#define	BLUEBOX_ISO18K6C_ACC_PWD_SIZE	(4)
/// ISO 18000-6C tag's kill password size in bytes.
#define	BLUEBOX_ISO18K6C_KILL_PWD_SIZE	(4)

/// ISO18000-6B tag's memory bank.
typedef enum BLUEBOX_ISO18K6C_Bank
{
	/// Reserved.
	BLUEBOX_ISO18K6C_BANK_RESERVED = 0,
	/// EPC.
	BLUEBOX_ISO18K6C_BANK_EPC = 1,
	/// TID.
	BLUEBOX_ISO18K6C_BANK_TID = 2,
	/// User memory.
	BLUEBOX_ISO18K6C_BANK_USER = 3

} BLUEBOX_ISO18K6C_Bank;

/// The ISO 18000-6C tag password permission values.
typedef enum BLUEBOX_ISO18K6C_PasswordPermission
{
	/// Accessible.
    BLUEBOX_ISO18K6C_TAG_PWD_PERM_ACCESSIBLE = 0,
	/// Always accessible.
    BLUEBOX_ISO18K6C_TAG_PWD_PERM_ALWAYS_ACCESSIBLE = 1,
	/// Secured accessible.
    BLUEBOX_ISO18K6C_TAG_PWD_PERM_SECURED_ACCESSIBLE = 2,
	/// Always not accessible.
    BLUEBOX_ISO18K6C_TAG_PWD_PERM_ALWAYS_NOT_ACCESSIBLE = 3,
	/// No change.
    BLUEBOX_ISO18K6C_TAG_PWD_PERM_NO_CHANGE = 4

} BLUEBOX_ISO18K6C_PasswordPermission;

/// The ISO 18000-6C tag memory permission values.
typedef enum BLUEBOX_ISO18K6C_MemoryPermission
{
	/// Accessible.
    BLUEBOX_ISO18K6C_TAG_MEM_PERM_WRITABLE = 0,
	/// Always accessible.
    BLUEBOX_ISO18K6C_TAG_MEM_PERM_ALWAYS_WRITABLE = 1,
	/// Secured accessible.
    BLUEBOX_ISO18K6C_TAG_MEM_PERM_SECURED_WRITABLE = 2,
	/// Always not accessible.
    BLUEBOX_ISO18K6C_TAG_MEM_PERM_ALWAYS_NOT_WRITABLE = 3,
	/// No change.
    BLUEBOX_ISO18K6C_TAG_MEM_PERM_NO_CHANGE = 4

} BLUEBOX_ISO18K6C_MemoryPermission;

/// ACTIVE tag's UID size in bytes.
#define	BLUEBOX_ACTIVE_UID_SIZE		(8)

/// Tag notification struct.
typedef struct BLUEBOX_Notify
{
	/// Reader address.
	unsigned char Address;
	/// Tag type.
	BLUEBOX_TagType TagType;
	/// Tag ID.
	unsigned char *Id;
	/// The length of the ID.
	int Length;
	/// The antenna used to identify the tag.
	BLUEBOX_Antenna Antenna;
	// The input which activates the identification procedure.
	BLUEBOX_Input Input;

} BLUEBOX_Notify;

/// Reader ID enum to use during upgrade or get firmware version.
typedef enum BLUEBOX_Reader
{
	/// Primary reader.
	BLUEBOX_PRIMARY_READER = 0,
	/// First auxiliary reader.
	BLUEBOX_AUXILIARY_1_READER = 1,
	/// Second auxiliary reader.
	BLUEBOX_AUXILIARY_2_READER = 2

} BLUEBOX_Reader;

/// Registration struct.
typedef struct BLUEBOX_Registration
{
	/// Tag type.
	BLUEBOX_TagType TagType;
	/// Tag ID.
	unsigned char * Id;
	/// The length of the ID.
	int Length;
	/// The antenna used to identify the tag.
	BLUEBOX_Antenna Antenna;
	// The input which activates the identification procedure.
	BLUEBOX_Input Input;

} BLUEBOX_Registration;

//-----------------------------------------------------------------------------
//--- PUBLIC VARIABLES --------------------------------------------------------
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
//--- PUBLIC FUNCTION PROTOTYPES ----------------------------------------------
//-----------------------------------------------------------------------------

BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_GetSwRelease (char *SwRel);

BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_Init (BLUEBOX_Handle *Handle);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_End (BLUEBOX_Handle *Handle);

BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_Open (BLUEBOX_Handle *Handle);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_Close (BLUEBOX_Handle *Handle);

BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_SetAddress (BLUEBOX_Handle *Handle, unsigned char Address);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_SetDevice (BLUEBOX_Handle *Handle, char *Type, char *Frequency, char *Range, char *Antennas, int Major, int Minor, char Variant);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_GetDevice (BLUEBOX_Handle *Handle, char *Type, char *Frequency, char *Range, char *Antennas, int *Major, int *Minor, char *Variant);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_SetChannel (BLUEBOX_Handle *Handle, char *Channel, char *Settings);

BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_GetFwRelease (BLUEBOX_Handle *Handle, BLUEBOX_Reader Reader, char *FwRel);

BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_Reset (BLUEBOX_Handle *Handle);

BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_GetDateTime (BLUEBOX_Handle *Handle, char *DateTime);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_SetDateTime (BLUEBOX_Handle *Handle, char *DateTime);

BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_ReadParameters (BLUEBOX_Handle *Handle, unsigned char *Parameters);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_WriteParameters (BLUEBOX_Handle *Handle, unsigned char *Parameters);

BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_DefaultParameters (BLUEBOX_Handle *Handle);

BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_ReadSerialNumber (BLUEBOX_Handle *Handle, unsigned char *Serial);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_ReadMACAddress (BLUEBOX_Handle *Handle, unsigned char *MAC);

BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_ReadConfiguration (BLUEBOX_Handle *Handle, int Page, unsigned char *Config);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_WriteConfiguration (BLUEBOX_Handle *Handle, int Page, unsigned char *Config);

BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_DefaultConfiguration (BLUEBOX_Handle *Handle);

BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_DataRequest (BLUEBOX_Handle *Handle, BLUEBOX_Tag **Tags, int *TagsNo);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_QueueRequest (BLUEBOX_Handle *Handle, BLUEBOX_Tag **Tags, int *TagsNo);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_FreeTagsMemory (BLUEBOX_Handle *Handle, BLUEBOX_Tag **Tags, int TagsNo);

BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_AllocateNotifyChannel (BLUEBOX_Handle *Handle, unsigned char Address);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_DeallocateNotifyChannel (BLUEBOX_Handle *Handle, unsigned char Address);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_GetNotification (BLUEBOX_Handle *Handle, BLUEBOX_Notify **Tags, int *TagsNo);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_FreeNotifyMemory (BLUEBOX_Handle *Handle, BLUEBOX_Notify **Tags, int TagsNo);

BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_SetOutput (BLUEBOX_Handle *Handle, BLUEBOX_Output Output, unsigned char Period);

BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_GetReaderStatus (BLUEBOX_Handle *Handle, unsigned short *Status);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_GetTemperature (BLUEBOX_Handle *Handle, double *Temperature);

BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_GetReflectedPower (BLUEBOX_Handle *Handle, int Frequency, int *IChannel, int *QChannel, double *Power);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_SelectiveGetReflectedPower (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, int Frequency, int *IChannel, int *QChannel, double *Power);

BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_RfOnOff (BLUEBOX_Handle *Handle, short OnOff);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_SelectiveRfOnOff (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, short OnOff);

BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_ReadID_EM4305 (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, BLUEBOX_TagType * TagType, void * Data);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_Write_EM4305 (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * Data, int Length);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_ReadID_T5557 (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, BLUEBOX_TagType * TagType, void * Data);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_Write_T5557 (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * Data, int Length);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_ReadID_Q5 (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, BLUEBOX_TagType * TagType, void * Data);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_Write_Q5 (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * Data, int Length);

BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_ReadID_HITAG1 (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, BLUEBOX_TagType * TagType, void * Data);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_ReadPage_HITAG1 (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * Id, int Page, void * Data);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_WritePage_HITAG1 (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * Id, int Page, void * Data);

BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_ReadID_HITAG2 (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, BLUEBOX_TagType * TagType, void * Data);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_Write_HITAG2 (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * Data, int Length);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_Login_HITAG2 (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * Id, void * Password);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_ReadPage_HITAG2 (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * Id, void * Password, int Page, void * Data);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_WritePage_HITAG2 (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * Id, void * Password, int Page, void * Data);

BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_ReadID_HITAGS (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, BLUEBOX_TagType * TagType, void * Data);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_Write_HITAGS (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * Data, int Length);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_ReadPage_HITAGS (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * Id, int Page, void * Data);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_WritePage_HITAGS (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * Id, int Page, void * Data);

BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_ReadID_TITAN (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, BLUEBOX_TagType * TagType, void * Data);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_Reset_TITAN (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_Login_TITAN (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * Password);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_WritePassword_TITAN (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * OldPwd, void * NewPwd);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_SelectiveRead_TITAN (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, int Address, int Length, void * Data);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_SelectiveWrite_TITAN (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, int Address, int Length, void * Data);

BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_Inventory_ISO15693 (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, short UseAFI, unsigned char AFI, BLUEBOX_Tag **Tags, int *TagsNo);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_ReadPage_ISO15693 (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * Id, int Page, int Size, void * Data);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_ReadMultiPage_ISO15693 (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * Id, int Page, int Size, int Count, void * Data);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_WritePage_ISO15693 (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * Id, int Page, int Size, void * Data);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_WriteMultiPage_ISO15693 (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * Id, int Page, int Size, int Count, void * Data);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_LockPage_ISO15693 (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * Id, int Page);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_GetSystemInfo_ISO15693 (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * Id, void * SysInfo);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_Write_AFI_ISO15693 (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * Id, unsigned char Afi);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_Lock_AFI_ISO15693 (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * Id);

BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_GetRandomNumber_ICODE_SLI_S (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * Id, void * Random);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_SetPassword_ICODE_SLI_S (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * Id, BLUEBOX_ICODE_SLI_S_PasswordIdentifier PwdId, void * Password, void * Random);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_WritePassword_ICODE_SLI_S (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * Id, BLUEBOX_ICODE_SLI_S_PasswordIdentifier PwdId, void * Password);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_LockPassword_ICODE_SLI_S (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * Id, BLUEBOX_ICODE_SLI_S_PasswordIdentifier PwdId);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_64BitPasswordProtection_ICODE_SLI_S (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * Id);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_ProtectPage_ICODE_SLI_S (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * Id, int PageNo, BLUEBOX_ICODE_SLI_S_ProtectionStatus Status);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_LockPageProtectionCondition_ICODE_SLI_S (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * Id, int PageNo);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_GetMultipleBlockProtectionStatus_ICODE_SLI_S (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * Id, int BlockNo, int Length, BLUEBOX_ICODE_SLI_S_BlockProtectionStatus * Status);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_Destroy_SLI_S_ICODE_SLI_S (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * Id);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_EnablePrivacy_ICODE_SLI_S (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * Id);

BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_Inventory_ISO14443A (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, BLUEBOX_Tag **Tags, int *TagsNo);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_ReadBlock_MIFARE_1k (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * Id, BLUEBOX_MIFARE_Key KeyType, void * Key, int Block, void * Data);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_WriteBlock_MIFARE_1k (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * Id, BLUEBOX_MIFARE_Key KeyType, void * Key, int Block, void * Data);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_ReadBlock_MIFARE_4k (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * Id, BLUEBOX_MIFARE_Key KeyType, void * Key, int Block, void * Data);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_WriteBlock_MIFARE_4k (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * Id, BLUEBOX_MIFARE_Key KeyType, void * Key, int Block, void * Data);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_ReadBlock_MIFARE_Ultralight (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * Id, int Block, void * Data);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_WriteBlock_MIFARE_Ultralight (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * Id, int Block, void * Data);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_ReadBlock_MIFARE_2k_7BUID (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * Id, BLUEBOX_MIFARE_Key KeyType, void * Key, int Block, void * Data);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_WriteBlock_MIFARE_2k_7BUID (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * Id, BLUEBOX_MIFARE_Key KeyType, void * Key, int Block, void * Data);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_ReadBlock_MIFARE_4k_7BUID (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * Id, BLUEBOX_MIFARE_Key KeyType, void * Key, int Block, void * Data);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_WriteBlock_MIFARE_4k_7BUID (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * Id, BLUEBOX_MIFARE_Key KeyType, void * Key, int Block, void * Data);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_ReadBlock_NTAG213 (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * Id, int Block, void * Data);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_WriteBlock_NTAG213 (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * Id, int Block, void * Data);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_ReadBlock_NTAG215 (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * Id, int Block, void * Data);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_WriteBlock_NTAG215 (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * Id, int Block, void * Data);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_ReadBlock_NTAG216 (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * Id, int Block, void * Data);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_WriteBlock_NTAG216 (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * Id, int Block, void * Data);

BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_Inventory_ISO14443B (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, BLUEBOX_Tag **Tags, int *TagsNo);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_ReadBlock_SR176 (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * Id, int Block, void * Data);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_WriteBlock_SR176 (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, void * Id, int Block, void * Data);

BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_Inventory_PICOPASS (BLUEBOX_Handle *Handle, BLUEBOX_Antenna Antenna, BLUEBOX_Tag **Tags, int *TagsNo);

BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_ReadRfParameters (BLUEBOX_Handle *Handle, unsigned char *Parameters);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_WriteRfParameters (BLUEBOX_Handle *Handle, unsigned char *Parameters);

BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_Inventory_ISO18K6B (BLUEBOX_Handle *Handle, BLUEBOX_Tag **Tags, int *TagsNo);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_Read_ISO18K6B (BLUEBOX_Handle *Handle, void *Uid, int Address, int NBlocks, void *Data);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_Write_ISO18K6B (BLUEBOX_Handle *Handle, void *Uid, int Address, int Length, void *Data);

BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_Inventory_ISO18K6C (BLUEBOX_Handle *Handle, BLUEBOX_Tag **Tags, int *TagsNo);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_Inventory_ISO18K6C_with_RSSI (BLUEBOX_Handle *Handle, BLUEBOX_Tag **Tags, int *TagsNo);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_ProgramEPC_ISO18K6C (BLUEBOX_Handle *Handle, void *Uid, void *Pwd, int Length, void *Data);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_Read_ISO18K6C (BLUEBOX_Handle *Handle, void *Uid, void *Pwd, BLUEBOX_ISO18K6C_Bank Bank, int Address, int Length, void *Data);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_Write_ISO18K6C (BLUEBOX_Handle *Handle, void *Uid, void *Pwd, BLUEBOX_ISO18K6C_Bank Bank, int Address, int Length, void *Data);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_BlockWrite_ISO18K6C (BLUEBOX_Handle *Handle, void *Uid, void *Pwd, BLUEBOX_ISO18K6C_Bank Bank, int Address, int Length, void *Data);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_Lock_ISO18K6C (BLUEBOX_Handle *Handle, void *Uid, void *Pwd, BLUEBOX_ISO18K6C_PasswordPermission KillPwd, BLUEBOX_ISO18K6C_PasswordPermission AccessPwd, BLUEBOX_ISO18K6C_MemoryPermission EPCMemory, BLUEBOX_ISO18K6C_MemoryPermission TIDMemory, BLUEBOX_ISO18K6C_MemoryPermission UserMemory);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_Kill_ISO18K6C (BLUEBOX_Handle *Handle, void *Uid, void *Pwd);

BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_FwUpgrade (BLUEBOX_Handle *Handle, BLUEBOX_Reader Reader, char *FileName);

BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_ReadNumberOfRegistrations (BLUEBOX_Handle *Handle, int *Registrations);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_ReadOlderRegistration (BLUEBOX_Handle *Handle, int *Index, BLUEBOX_Registration *Registration);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_CancelOlderRegistration (BLUEBOX_Handle *Handle, int Index);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_CancelAllRegistrations (BLUEBOX_Handle *Handle);
BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_ReadPreviousRegistration (BLUEBOX_Handle *Handle, int Index, BLUEBOX_Registration *Registration);

BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_GenericCommand (BLUEBOX_Handle *Handle, char * Command, char * Reply);

BLUEBOXLib_API BLUEBOX_ErrorCodes __stdcall BLUEBOX_KL_Notify (BLUEBOX_Handle *Handle, unsigned char Notify);

#ifdef __cplusplus
}
#endif

#endif
